#!/usr/bin/env python3
"""
Elevenlabs API Test Script
Bu script, Elevenlabs API'sinin düzgün çalışıp çalışmadığını test eder.
"""

import os
import sys
from elevenlabs import ElevenLabs

def test_elevenlabs_api():
    """Test Elevenlabs API functionality"""
    
    # API anahtarını al
    api_key = os.getenv('ELEVENLABS_API_KEY')
    
    if not api_key:
        print("❌ ELEVENLABS_API_KEY çevre değişkeni ayarlanmamış!")
        print("Kullanım: ELEVENLABS_API_KEY=your_key python test_elevenlabs.py")
        return False
    
    try:
        # ElevenLabs client'ı oluştur
        client = ElevenLabs(api_key=api_key)
        
        print("🔄 Elevenlabs API bağlantısı test ediliyor...")
        
        # Sesler listesini al
        print("📋 Mevcut sesler getiriliyor...")
        voices = client.voices.get_all()
        
        if voices.voices:
            print(f"✅ {len(voices.voices)} adet ses bulundu:")
            for i, voice in enumerate(voices.voices[:5]):  # İlk 5 sesi göster
                print(f"   {i+1}. {voice.name} ({voice.voice_id})")
            
            if len(voices.voices) > 5:
                print(f"   ... ve {len(voices.voices) - 5} ses daha")
        else:
            print("⚠️  Hiç ses bulunamadı")
            return False
        
        # Test metni sese çevir
        print("\n🔊 Text-to-Speech testi yapılıyor...")
        test_text = "Merhaba! Bu bir Elevenlabs API test mesajıdır. Türkçe karakterler: ğüşıöç"
        
        # İlk sesi kullanarak TTS test et
        first_voice = voices.voices[0]
        print(f"   Kullanılan ses: {first_voice.name}")
        
        # Ses dosyası oluştur
        audio_generator = client.text_to_speech.convert(
            voice_id=first_voice.voice_id,
            text=test_text,
            model_id="eleven_multilingual_v2"  # Türkçe desteği için
        )
        
        # Ses dosyasını kaydet
        output_file = "test_output.mp3"
        with open(output_file, "wb") as f:
            for chunk in audio_generator:
                f.write(chunk)
        
        print(f"✅ Ses dosyası başarıyla oluşturuldu: {output_file}")
        print(f"   Dosya boyutu: {os.path.getsize(output_file)} bytes")
        
        return True
        
    except Exception as e:
        print(f"❌ API testi başarısız: {str(e)}")
        return False

def main():
    """Ana fonksiyon"""
    print("🚀 Elevenlabs API Test Aracı")
    print("=" * 40)
    
    if test_elevenlabs_api():
        print("\n🎉 Tüm testler başarılı!")
        print("✨ PHP uygulamanız Elevenlabs API'si ile çalışmaya hazır.")
    else:
        print("\n💥 Test başarısız!")
        print("🔧 API anahtarınızı ve internet bağlantınızı kontrol edin.")
        sys.exit(1)

if __name__ == "__main__":
    main() 