<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $config['app']['name']; ?> - Ana Panel</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .call-simulation {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        .pulse-ring {
            animation: pulse-ring 1.25s cubic-bezier(0.215, 0.61, 0.355, 1) infinite;
        }
        @keyframes pulse-ring {
            0% { transform: scale(.33); }
            80%, 100% { transform: scale(1.33); opacity: 0; }
        }
        .status-indicator {
            display: inline-block;
            width: 8px;
            height: 8px;
            border-radius: 50%;
            margin-right: 8px;
        }
        .status-success { background-color: #10b981; }
        .status-warning { background-color: #f59e0b; }
        .status-error { background-color: #ef4444; }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    <!-- Header -->
    <header class="bg-white shadow-sm border-b">
        <div class="container mx-auto px-4 py-4">
            <div class="flex items-center justify-between">
                <div class="flex items-center space-x-3">
                    <i class="fas fa-home text-blue-600 text-2xl"></i>
                    <h1 class="text-2xl font-bold text-gray-800"><?php echo $config['app']['name']; ?></h1>
                    <span class="bg-blue-100 text-blue-800 text-xs px-2 py-1 rounded-full">v<?php echo $config['app']['version']; ?></span>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-sm text-gray-600">
                        <i class="fas fa-robot mr-1"></i>
                        <strong><?php echo $config['assistant']['name']; ?></strong>
                        <span class="text-gray-400">• <?php echo $config['assistant']['company']; ?></span>
                    </span>
                    <a href="/admin" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                        <i class="fas fa-cog mr-2"></i>Yönetim
                    </a>
                </div>
            </div>
        </div>
    </header>

    <div class="container mx-auto px-4 py-8">
        <!-- System Status Alert -->
        <div id="system-status-alert" class="mb-6 hidden">
            <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 rounded">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation-triangle text-yellow-400"></i>
                    </div>
                    <div class="ml-3">
                        <p class="text-sm text-yellow-700" id="status-message">
                            Sistem durumu kontrol ediliyor...
                        </p>
                    </div>
                </div>
            </div>
        </div>

        <div class="grid lg:grid-cols-2 gap-8">
            <!-- Test Panel -->
            <div class="bg-white rounded-lg shadow-lg overflow-hidden">
                <div class="call-simulation text-white p-6">
                    <h2 class="text-2xl font-bold mb-2">
                        <i class="fas fa-phone-alt mr-2"></i>
                        AI Asistan Test
                    </h2>
                    <p class="opacity-90">Gayrimenkul AI asistanınızı test edin</p>
                </div>
                
                <div class="p-6">
                    <!-- Customer Info Form -->
                    <div class="mb-6">
                        <h3 class="text-lg font-semibold mb-4">📋 Müşteri Bilgileri</h3>
                        <div class="grid md:grid-cols-2 gap-4 mb-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Ad Soyad</label>
                                <input type="text" id="customer-name" placeholder="Ahmet Kaya" value="Ahmet Özkan"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Telefon</label>
                                <input type="tel" id="customer-phone" placeholder="05xx xxx xx xx" value="05322191827"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                                       onblur="formatPhoneNumber(this)">
                            </div>
                        </div>
                        
                        <!-- Customer Context/Pre-Info -->
                        <div class="bg-blue-50 p-4 rounded-lg">
                            <h4 class="text-md font-semibold text-blue-800 mb-3">🎯 Müşteri Ön Bilgileri (AI'ya Aktarılacak)</h4>
                            <div class="grid md:grid-cols-2 gap-4 mb-3">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">İlgi Alanı</label>
                                    <select id="customer-interest" class="w-full px-2 py-1.5 border border-gray-300 rounded text-sm">
                                        <option value="">Seçiniz</option>
                                        <option value="2+1 daire">2+1 Daire</option>
                                        <option value="3+1 daire" selected>3+1 Daire</option>
                                        <option value="4+1 daire">4+1 Daire</option>
                                        <option value="villa">Villa</option>
                                        <option value="ofis">Ofis/Ticari</option>
                                        <option value="arsa">Arsa</option>
                                        <option value="yatırım">Yatırım Amaçlı</option>
                                    </select>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">Bütçe Aralığı</label>
                                    <select id="customer-budget" class="w-full px-2 py-1.5 border border-gray-300 rounded text-sm">
                                        <option value="">Seçiniz</option>
                                        <option value="500k-1M">500k - 1M TL</option>
                                        <option value="1M-2M" selected>1M - 2M TL</option>
                                        <option value="2M-5M">2M - 5M TL</option>
                                        <option value="5M+">5M+ TL</option>
                                    </select>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">Tercih Edilen Bölge</label>
                                    <input type="text" id="customer-location" placeholder="Örn: Kadıköy, Beşiktaş..." value="Kadıköy, Beşiktaş"
                                           class="w-full px-2 py-1.5 border border-gray-300 rounded text-sm">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1">Aciliyet</label>
                                    <select id="customer-urgency" class="w-full px-2 py-1.5 border border-gray-300 rounded text-sm">
                                        <option value="">Seçiniz</option>
                                        <option value="acil">Acil (1 hafta)</option>
                                        <option value="normal" selected>Normal (1 ay)</option>
                                        <option value="flexible">Esnek (3+ ay)</option>
                                    </select>
                                </div>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">Ek Notlar</label>
                                <textarea id="customer-notes" placeholder="Özel istekler, ek bilgiler..." 
                                          class="w-full px-2 py-1.5 border border-gray-300 rounded text-sm" rows="2">Deniz manzaralı tercih ediyor, yatırım amaçlı</textarea>
                            </div>
                        </div>
                    </div>

                    <!-- Call Simulation -->
                    <div class="text-center">
                        <div id="call-status" class="mb-4">
                            <div class="inline-flex items-center justify-center w-20 h-20 bg-green-100 rounded-full mb-3">
                                <i class="fas fa-phone text-green-600 text-2xl"></i>
                            </div>
                            <p class="text-gray-600">Aramayı başlatmak için butona tıklayın</p>
                        </div>
                        
                        <button id="start-call-btn" onclick="startCallSimulation()" 
                                class="px-8 py-3 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors">
                            <i class="fas fa-phone mr-2"></i>Aramayı Başlat
                        </button>
                        
                        <button id="end-call-btn" onclick="endCallSimulation()" 
                                class="px-8 py-3 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors hidden">
                            <i class="fas fa-phone-slash mr-2"></i>Aramayı Sonlandır
                        </button>
                    </div>

                    <!-- Chat Interface -->
                    <div id="chat-interface" class="hidden mt-6">
                        <div class="bg-blue-50 border border-blue-200 rounded-lg p-3 mb-4">
                            <div class="flex items-center">
                                <i class="fas fa-info-circle text-blue-600 mr-2"></i>
                                <div class="text-sm text-blue-800">
                                    <strong>Konuşma Kaydı:</strong> Arama bittiğinde buraya gerçek transcript gelecek
                                </div>
                            </div>
                        </div>
                        
                        <div class="border rounded-lg p-4 h-64 overflow-y-auto mb-4 bg-gray-50" id="chat-messages">
                            <!-- Messages will appear here -->
                        </div>
                        
                        <div class="flex gap-2">
                            <input type="text" id="message-input" placeholder="📝 Notlarınızı yazabilirsiniz (sadece görüntüleme amaçlı)..." 
                                   class="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                                   onkeypress="handleKeyPress(event)">
                            <button onclick="sendMessage()" 
                                    class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                                <i class="fas fa-paper-plane"></i>
                            </button>
                        </div>
                        
                        <div class="text-xs text-gray-500 mt-2">
                            💡 <strong>Bilgi:</strong> Bu alan gerçek telefon araması sırasında transcript göstermek için kullanılır. 
                            Mesaj gönderme simülasyon amaçlıdır.
                        </div>
                    </div>
                </div>
            </div>

            <!-- System Info & Controls -->
            <div class="space-y-6">
                <!-- System Status -->
                <div class="bg-white rounded-lg shadow-lg p-6">
                    <h3 class="text-lg font-semibold mb-4">
                        <i class="fas fa-server text-blue-600 mr-2"></i>
                        Sistem Durumu
                    </h3>
                    
                    <div class="space-y-3">
                        <div class="flex items-center justify-between">
                            <span class="text-gray-600">API Bağlantısı</span>
                            <span id="api-status" class="px-2 py-1 text-xs rounded-full bg-yellow-100 text-yellow-800">
                                <span class="status-indicator bg-yellow-400"></span>
                                Kontrol ediliyor...
                            </span>
                        </div>
                        
                        <div class="flex items-center justify-between">
                            <span class="text-gray-600">Agent ID</span>
                            <span class="text-xs font-mono bg-gray-100 px-2 py-1 rounded">
                                <?php echo substr($config['elevenlabs']['agent_id'], 0, 8); ?>...
                            </span>
                        </div>
                        
                        <div class="flex items-center justify-between">
                            <span class="text-gray-600">Telefon Numarası</span>
                            <span class="text-xs font-mono bg-gray-100 px-2 py-1 rounded">
                                <?php echo substr($config['twilio']['phone_number_id'], 0, 8); ?>...
                            </span>
                        </div>
                        
                        <div class="flex items-center justify-between">
                            <span class="text-gray-600">Bilgi Tabanı</span>
                            <span id="knowledge-count" class="px-2 py-1 text-xs rounded-full bg-blue-100 text-blue-800">
                                Yükleniyor...
                            </span>
                        </div>
                    </div>
                </div>

                <!-- AI Assistant Info -->
                <div class="bg-white rounded-lg shadow-lg p-6">
                    <h3 class="text-lg font-semibold mb-4">
                        <i class="fas fa-robot text-purple-600 mr-2"></i>
                        AI Asistan Bilgileri
                    </h3>
                    
                    <div class="space-y-3">
                        <div class="flex items-center justify-between">
                            <span class="text-gray-600">İsim</span>
                            <span class="font-medium"><?php echo $config['assistant']['name']; ?></span>
                        </div>
                        
                        <div class="flex items-center justify-between">
                            <span class="text-gray-600">Rol</span>
                            <span class="font-medium"><?php echo $config['assistant']['role']; ?></span>
                        </div>
                        
                        <div class="flex items-center justify-between">
                            <span class="text-gray-600">Şirket</span>
                            <span class="font-medium"><?php echo $config['assistant']['company']; ?></span>
                        </div>
                        
                        <div class="flex items-center justify-between">
                            <span class="text-gray-600">Model</span>
                            <span class="text-xs bg-green-100 text-green-800 px-2 py-1 rounded">
                                <?php echo $config['elevenlabs']['model']; ?>
                            </span>
                        </div>
                    </div>
                </div>

                <!-- Voice Selection -->
                <div class="bg-white rounded-lg shadow-lg p-6">
                    <h3 class="text-lg font-semibold mb-4">
                        <i class="fas fa-microphone text-purple-500 mr-2"></i>
                        Ses Seçimi
                    </h3>
                    
                    <div class="space-y-3">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">AI Asistan Sesi</label>
                            <select id="voice-selector" 
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                                    onchange="selectVoice()">
                                <option value="">Yükleniyor...</option>
                            </select>
                        </div>
                        
                        <div id="voice-info" class="hidden">
                            <div class="text-sm text-gray-600 bg-gray-50 p-3 rounded">
                                <div class="flex items-center justify-between">
                                    <span>Dil:</span>
                                    <span id="voice-language" class="font-medium">-</span>
                                </div>
                                <div class="flex items-center justify-between mt-1">
                                    <span>Kategori:</span>
                                    <span id="voice-category" class="font-medium">-</span>
                                </div>
                            </div>
                        </div>
                        
                        <button onclick="testVoice()" 
                                class="w-full px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors">
                            <i class="fas fa-play mr-2"></i>
                            Sesi Test Et
                        </button>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="bg-white rounded-lg shadow-lg p-6">
                    <h3 class="text-lg font-semibold mb-4">
                        <i class="fas fa-lightning-bolt text-yellow-500 mr-2"></i>
                        Hızlı İşlemler
                    </h3>
                    
                    <div class="space-y-3">
                        <button onclick="testAPIConnection()" 
                                class="w-full px-4 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-left">
                            <i class="fas fa-plug mr-2"></i>
                            Sistem Bağlantısını Test Et
                        </button>
                        
                        <button onclick="initializeKnowledgeBase()" 
                                class="w-full px-4 py-3 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors text-left">
                            <i class="fas fa-database mr-2"></i>
                            Varsayılan Bilgi Tabanını Yükle
                        </button>
                        
                        <button onclick="showKnowledgeBaseModal()" 
                                class="w-full px-4 py-3 bg-orange-600 text-white rounded-lg hover:bg-orange-700 transition-colors text-left">
                            <i class="fas fa-books mr-2"></i>
                            Bilgi Veritabanı Yönetimi
                        </button>
                        
                        <a href="/admin" 
                           class="block w-full px-4 py-3 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition-colors text-left">
                            <i class="fas fa-cogs mr-2"></i>
                            Gelişmiş Ayarlar
                        </a>
                    </div>
                </div>

                <!-- Recent Activity -->
                <div class="bg-white rounded-lg shadow-lg p-6">
                    <h3 class="text-lg font-semibold mb-4">
                        <i class="fas fa-clock text-gray-600 mr-2"></i>
                        Son Aktiviteler
                    </h3>
                    
                    <div id="recent-activity" class="space-y-3 text-sm text-gray-600">
                        <div class="flex items-center">
                            <i class="fas fa-circle text-blue-400 text-xs mr-2"></i>
                            Sistem başlatıldı
                            <span class="text-xs text-gray-400 ml-auto"><?php echo date('H:i'); ?></span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Knowledge Base Modal -->
    <div id="knowledge-base-modal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen px-4">
            <div class="bg-white rounded-lg shadow-xl w-full max-w-6xl max-h-[90vh] overflow-hidden">
                <div class="flex items-center justify-between p-6 border-b">
                    <h3 class="text-xl font-semibold">
                        <i class="fas fa-books text-orange-600 mr-2"></i>
                        Bilgi Veritabanı Yönetimi
                    </h3>
                    <button onclick="hideKnowledgeBaseModal()" class="text-gray-400 hover:text-gray-600">
                        <i class="fas fa-times text-xl"></i>
                    </button>
                </div>
                
                <div class="flex h-[70vh]">
                    <!-- Sidebar -->
                    <div class="w-1/3 bg-gray-50 border-r overflow-y-auto">
                        <div class="p-4">
                            <div class="mb-4">
                                <button onclick="showAddContentForm()" 
                                        class="w-full px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors">
                                    <i class="fas fa-plus mr-2"></i>
                                    Yeni İçerik Ekle
                                </button>
                            </div>
                            
                            <div class="mb-4">
                                <label class="block text-sm font-medium text-gray-700 mb-2">Kategori Filtresi</label>
                                <select id="category-filter" onchange="filterKnowledgeBase()" 
                                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    <option value="">Tüm Kategoriler</option>
                                </select>
                            </div>
                            
                            <div class="mb-4">
                                <label class="block text-sm font-medium text-gray-700 mb-2">Arama</label>
                                <input type="text" id="knowledge-search" placeholder="İçerikte ara..." 
                                       onkeyup="searchKnowledgeBase()"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            
                            <div id="knowledge-categories" class="space-y-2">
                                <!-- Categories will be loaded here -->
                            </div>
                        </div>
                    </div>
                    
                    <!-- Main Content -->
                    <div class="w-2/3 flex flex-col">
                        <div class="flex-1 overflow-y-auto p-6">
                            <div id="knowledge-content-list">
                                <!-- Content list will be loaded here -->
                            </div>
                            
                            <!-- Add Content Form -->
                            <div id="add-content-form" class="hidden">
                                <h4 class="text-lg font-semibold mb-4">Yeni İçerik Ekle</h4>
                                <form onsubmit="addKnowledgeContent(event)" class="space-y-4">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Başlık</label>
                                        <input type="text" id="content-title" required 
                                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    </div>
                                    
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Kategori</label>
                                        <select id="content-category" required 
                                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                                            <option value="">Kategori Seçin</option>
                                        </select>
                                    </div>
                                    
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">İçerik</label>
                                        <textarea id="content-text" rows="8" required 
                                                  class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                                                  placeholder="İçerik metnini buraya yazın..."></textarea>
                                    </div>
                                    
                                    <div class="flex space-x-2">
                                        <button type="submit" 
                                                class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors">
                                            <i class="fas fa-save mr-2"></i>
                                            Kaydet
                                        </button>
                                        <button type="button" onclick="hideAddContentForm()" 
                                                class="px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700 transition-colors">
                                            İptal
                                        </button>
                                    </div>
                                </form>
                            </div>
                            
                            <!-- Edit Content Form -->
                            <div id="edit-content-form" class="hidden">
                                <h4 class="text-lg font-semibold mb-4">İçerik Düzenle</h4>
                                <form onsubmit="updateKnowledgeContent(event)" class="space-y-4">
                                    <input type="hidden" id="edit-content-id">
                                    
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Başlık</label>
                                        <input type="text" id="edit-content-title" required 
                                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    </div>
                                    
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">İçerik</label>
                                        <textarea id="edit-content-text" rows="8" required 
                                                  class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"></textarea>
                                    </div>
                                    
                                    <div class="flex space-x-2">
                                        <button type="submit" 
                                                class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                                            <i class="fas fa-save mr-2"></i>
                                            Güncelle
                                        </button>
                                        <button type="button" onclick="hideEditContentForm()" 
                                                class="px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700 transition-colors">
                                            İptal
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        let conversationId = null;
        let isCallActive = false;
        let callId = null;
        let statusCheckInterval = null;

        // Format phone number for Turkish numbers
        function formatPhoneNumber(input) {
            let phone = input.value.trim();
            
            if (!phone) return;
            
            // Remove all non-numeric characters except +
            let cleaned = phone.replace(/[^\d+]/g, '');
            
            // Handle Turkish phone number formats
            if (cleaned.startsWith('0')) {
                // If starts with 0, convert to +90
                cleaned = '+90' + cleaned.substring(1);
            } else if (cleaned.startsWith('90') && !cleaned.startsWith('+')) {
                // If starts with 90 but no +, add +
                cleaned = '+' + cleaned;
            } else if (cleaned.match(/^\d{10}$/)) {
                // If 10 digits without country code, assume Turkish
                cleaned = '+90' + cleaned;
            } else if (!cleaned.startsWith('+')) {
                // If doesn't start with +, add +90 if looks like Turkish number
                if (cleaned.length >= 10) {
                    cleaned = '+90' + cleaned;
                }
            }
            
            // Update input value
            input.value = cleaned;
            
            // Visual feedback
            if (cleaned.match(/^\+90[0-9]{10}$/)) {
                input.style.borderColor = '#10b981'; // green
                input.style.backgroundColor = '#f0fdf4'; // light green
            } else {
                input.style.borderColor = '#f59e0b'; // amber
                input.style.backgroundColor = '#fffbeb'; // light amber
            }
        }

        // Initialize page
        document.addEventListener('DOMContentLoaded', function() {
            testAPIConnection();
            loadKnowledgeBaseStatus();
            loadVoiceList();
        });

        // Test API connection
        async function testAPIConnection() {
            try {
                document.getElementById('api-status').innerHTML = 
                    '<span class="status-indicator bg-yellow-400"></span>Test ediliyor...';
                
                const response = await fetch('/api/test');
                const data = await response.json();
                
                if (data.success) {
                    document.getElementById('api-status').innerHTML = 
                        '<span class="status-indicator status-success"></span>Bağlı ✓';
                    addActivity('✅ ' + data.summary);
                    
                    // Show detailed results in console
                    console.log('🔍 Detaylı Test Sonuçları:', data);
                    
                    // Update UI with test results
                    if (data.tests.api_connection && data.tests.api_connection.voice_count) {
                        addActivity(`🎤 ${data.tests.api_connection.voice_count} ses seçeneği bulundu`);
                    }
                    
                    // Show success message for mostly_success status
                    if (data.overall_status === 'mostly_success') {
                        showSystemAlert('Sistem başarıyla çalışıyor! Sadece ses seçimi uyarısı var, bu normal.', 'success');
                        setTimeout(hideSystemAlert, 5000);
                    } else {
                        hideSystemAlert();
                    }
                } else {
                    document.getElementById('api-status').innerHTML = 
                        '<span class="status-indicator status-error"></span>Bağlantı hatası';
                    addActivity('❌ API test hatası: ' + (data.error || 'Bilinmeyen hata'));
                    showSystemAlert('API bağlantısı kurulamadı. Lütfen API anahtarlarınızı kontrol edin.', 'error');
                }
                
            } catch (error) {
                document.getElementById('api-status').innerHTML = 
                    '<span class="status-indicator status-error"></span>Test başarısız';
                addActivity('🔌 Bağlantı hatası: ' + error.message);
                showSystemAlert('Sistem testi başarısız. Sunucu bağlantısını kontrol edin.', 'error');
            }
        }

        // Collect customer context information
        function getCustomerContext() {
            return {
                interest: document.getElementById('customer-interest').value,
                budget: document.getElementById('customer-budget').value,
                location: document.getElementById('customer-location').value,
                urgency: document.getElementById('customer-urgency').value,
                notes: document.getElementById('customer-notes').value
            };
        }

        // Start call simulation with context
        async function startCallSimulation() {
            const customerName = document.getElementById('customer-name').value.trim();
            let customerPhone = document.getElementById('customer-phone').value.trim();
            
            if (!customerName) {
                alert('Lütfen müşteri adını girin');
                return;
            }
            
            if (!customerPhone) {
                alert('Lütfen müşteri telefon numarasını girin (Gerçek telefon araması başlatılacak)');
                return;
            }
            
            // Telefon numarasını formatla
            const phoneInput = document.getElementById('customer-phone');
            formatPhoneNumber(phoneInput);
            customerPhone = phoneInput.value;
            
            // Türk telefon numarası formatını kontrol et
            if (!customerPhone.match(/^\+90[0-9]{10}$/)) {
                alert('Lütfen geçerli bir Türk telefon numarası girin.\n\nÖrnek formatlar:\n• +90 532 219 1827\n• 0532 219 1827\n• 5322191827');
                return;
            }
            
            // Get customer context
            const customerContext = getCustomerContext();
            
            // Create confirmation message with context
            let confirmMsg = `${customerName} (${customerPhone}) numarasını GERÇEKTEN aramak istediğinizden emin misiniz?\n\nBu işlem gerçek bir telefon araması başlatacaktır.`;
            
            if (customerContext.interest || customerContext.budget || customerContext.location) {
                confirmMsg += '\n\nAI asistanına şu bilgiler aktarılacak:';
                if (customerContext.interest) confirmMsg += `\n• İlgi Alanı: ${customerContext.interest}`;
                if (customerContext.budget) confirmMsg += `\n• Bütçe: ${customerContext.budget}`;
                if (customerContext.location) confirmMsg += `\n• Bölge: ${customerContext.location}`;
                if (customerContext.urgency) confirmMsg += `\n• Aciliyet: ${customerContext.urgency}`;
            }
            
            if (!confirm(confirmMsg)) {
                return;
            }

            try {
                isCallActive = true;
                updateCallUI(true);
                
                // Start conversation with context
                const response = await fetch('/api/conversation/start', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        customer_name: customerName,
                        customer_phone: customerPhone,
                        customer_context: customerContext
                    })
                });

                const data = await response.json();
                
                if (data.success) {
                    conversationId = data.conversation_id;
                    callId = data.call_id;
                    showChatInterface();
                    addMessage('Sistem', `✅ ${data.message || 'Telefon araması başlatıldı'}`);
                    addMessage('Asistan', data.greeting || `Merhaba ${customerName}, ben Ayşe Kaya. Size nasıl yardımcı olabilirim?`);
                    addActivity(`📞 GERÇEK ARAMA başlatıldı: ${customerName} (${customerPhone})`);
                    
                    // Gerçek zamanlı durum takibini başlat
                    startStatusChecking();
                    
                    // Start real-time monitoring
                    startRealTimeMonitoring();
                    
                    // Show success message
                    alert(`✅ Telefon araması başlatıldı!\n\n${customerPhone} numarasını arayacağız.\nKonuşma ID: ${data.conversation_id}`);
                } else {
                    throw new Error(data.error || 'Arama başlatılamadı');
                }
                
            } catch (error) {
                console.error('Call start error:', error);
                alert('Arama başlatılamadı: ' + error.message);
                endCallSimulation();
            }
        }

        // End call simulation
        function endCallSimulation() {
            isCallActive = false;
            updateCallUI(false);
            hideChatInterface();
            conversationId = null;
            callId = null;
            
            // Durum kontrolünü durdur
            if (statusCheckInterval) {
                clearInterval(statusCheckInterval);
                statusCheckInterval = null;
            }
            
            // Real-time monitoring'i durdur
            if (monitoringInterval) {
                clearInterval(monitoringInterval);
                monitoringInterval = null;
            }
            
            addActivity('📞 Arama ve monitoring sonlandırıldı');
        }

        // Update call UI
        function updateCallUI(active) {
            const startBtn = document.getElementById('start-call-btn');
            const endBtn = document.getElementById('end-call-btn');
            const status = document.getElementById('call-status');
            
            if (active) {
                startBtn.classList.add('hidden');
                endBtn.classList.remove('hidden');
                status.innerHTML = `
                    <div class="inline-flex items-center justify-center w-20 h-20 bg-green-100 rounded-full mb-3 relative">
                        <div class="absolute inset-0 rounded-full bg-green-400 opacity-25 pulse-ring"></div>
                        <i class="fas fa-phone text-green-600 text-2xl"></i>
                    </div>
                    <p class="text-green-600 font-medium">🔴 Arama aktif</p>
                `;
            } else {
                startBtn.classList.remove('hidden');
                endBtn.classList.add('hidden');
                status.innerHTML = `
                    <div class="inline-flex items-center justify-center w-20 h-20 bg-green-100 rounded-full mb-3">
                        <i class="fas fa-phone text-green-600 text-2xl"></i>
                    </div>
                    <p class="text-gray-600">Aramayı başlatmak için butona tıklayın</p>
                `;
            }
        }

        // Show/hide chat interface
        function showChatInterface() {
            document.getElementById('chat-interface').classList.remove('hidden');
            document.getElementById('chat-messages').innerHTML = '';
        }

        function hideChatInterface() {
            document.getElementById('chat-interface').classList.add('hidden');
        }

        // Add message to chat
        function addMessage(sender, message, customClass = null) {
            const chatMessages = document.getElementById('chat-messages');
            const messageDiv = document.createElement('div');
            messageDiv.className = `mb-3 ${sender === 'Siz' ? 'text-right' : 'text-left'}`;
            
            let bubbleClass;
            if (customClass) {
                bubbleClass = `${customClass} text-gray-800 mr-auto`;
            } else {
                bubbleClass = sender === 'Siz' 
                    ? 'bg-blue-600 text-white ml-auto' 
                    : 'bg-gray-200 text-gray-800 mr-auto';
            }
            
            messageDiv.innerHTML = `
                <div class="inline-block max-w-xs px-3 py-2 rounded-lg ${bubbleClass}">
                    <div class="font-medium text-xs mb-1">${sender}</div>
                    <div>${message}</div>
                </div>
                <div class="text-xs text-gray-500 mt-1 ${sender === 'Siz' ? 'text-right' : 'text-left'}">
                    ${new Date().toLocaleTimeString('tr-TR')}
                </div>
            `;
            
            chatMessages.appendChild(messageDiv);
            chatMessages.scrollTop = chatMessages.scrollHeight;
        }

        // Send message
        async function sendMessage() {
            const input = document.getElementById('message-input');
            const message = input.value.trim();
            
            if (!message || !isCallActive) return;
            
            addMessage('Siz', message);
            input.value = '';
            
            // Show typing indicator
            addMessage('Asistan', '⌨️ Yazıyor...');
            
            try {
                const response = await fetch('/api/conversation/send', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        conversation_id: conversationId,
                        message: message,
                        customer_info: {
                            name: document.getElementById('customer-name').value,
                            phone: document.getElementById('customer-phone').value
                        }
                    })
                });

                const data = await response.json();
                
                // Remove typing indicator
                const chatMessages = document.getElementById('chat-messages');
                if (chatMessages.lastChild) {
                    chatMessages.removeChild(chatMessages.lastChild);
                }
                
                if (data.success) {
                    addMessage('Asistan', data.response);
                    addActivity(`💬 Mesaj işlendi: "${message.substring(0, 30)}..."`);
                } else {
                    addMessage('Sistem', '⚠️ Mesaj gönderilemedi: ' + data.error);
                }
                
            } catch (error) {
                // Remove typing indicator
                const chatMessages = document.getElementById('chat-messages');
                if (chatMessages.lastChild) {
                    chatMessages.removeChild(chatMessages.lastChild);
                }
                
                addMessage('Sistem', '❌ Bağlantı hatası: ' + error.message);
                addActivity('⚠️ Mesaj gönderme hatası');
            }
        }

        // Handle Enter key
        function handleKeyPress(event) {
            if (event.key === 'Enter') {
                sendMessage();
            }
        }

        // Gerçek zamanlı durum takibi başlat
        function startStatusChecking() {
            if (!callId) return;
            
            // İlk kontrolü hemen yap
            checkCallStatus();
            
            // Her 10 saniyede bir kontrol et
            statusCheckInterval = setInterval(checkCallStatus, 10000);
        }

        // Real-time monitoring with detailed information
        let monitoringInterval = null;
        
        function startRealTimeMonitoring() {
            if (monitoringInterval) {
                clearInterval(monitoringInterval);
            }
            
            if (!callId) return;
            
            monitoringInterval = setInterval(async () => {
                if (!callId || !isCallActive) {
                    clearInterval(monitoringInterval);
                    return;
                }
                
                try {
                    const response = await fetch(`/api/conversation?action=monitor&call_id=${callId}`);
                    const data = await response.json();
                    
                    if (data.success && data.monitoring) {
                        updateRealTimeMonitoring(data.monitoring);
                    }
                } catch (error) {
                    console.error('Real-time monitoring error:', error);
                }
            }, 5000); // Monitor every 5 seconds
        }
        
        function updateRealTimeMonitoring(monitoring) {
            console.log('🔄 Real-time monitoring data:', monitoring);
            
            // Update call status with more details
            if (monitoring.status) {
                updateCallStatusUI(monitoring.status, monitoring);
            }
            
            // Show real-time transcript if available
            if (monitoring.live_transcript) {
                displayLiveTranscript(monitoring.live_transcript);
            }
            
            // Add activity log with conversation IDs
            if (monitoring.conversation_ids && monitoring.conversation_ids.length > 0) {
                addActivity(`🔄 Monitoring: ${monitoring.status} | Conversations: ${monitoring.conversation_ids.length}`);
            } else {
                addActivity(`🔄 Monitoring: ${monitoring.status} (${new Date(monitoring.timestamp * 1000).toLocaleTimeString()})`);
            }
            
            // Stop monitoring if call completed
            if (monitoring.status === 'completed' || monitoring.status === 'done') {
                clearInterval(monitoringInterval);
                addActivity('✅ Real-time monitoring tamamlandı');
            }
        }
        
        function displayLiveTranscript(transcript) {
            try {
                if (transcript && transcript.transcript) {
                    // Clear old messages and show real transcript
                    const chatMessages = document.getElementById('chat-messages');
                    chatMessages.innerHTML = '<div class="text-sm text-green-600 mb-2"><i class="fas fa-check-circle mr-1"></i><strong>🎯 Gerçek Konuşma Transkripti:</strong></div>';
                    
                    transcript.transcript.forEach(item => {
                        if (item.content) {
                            const role = item.role === 'agent' ? 'AI Asistan' : 'Müşteri';
                            const roleClass = item.role === 'agent' ? 'bg-blue-100' : 'bg-gray-100';
                            
                            addMessage(role, item.content, roleClass);
                        }
                    });
                    
                    // Show call summary if available
                    if (transcript.summary) {
                        addMessage('Sistem', `📋 Özet: ${transcript.summary}`, 'bg-yellow-100');
                    }
                    
                    if (transcript.duration_seconds) {
                        addMessage('Sistem', `⏱️ Süre: ${transcript.duration_seconds} saniye`, 'bg-green-100');
                    }
                    
                    if (transcript.cost_breakdown && transcript.cost_breakdown.total_cost_cents) {
                        addMessage('Sistem', `💰 Maliyet: ${transcript.cost_breakdown.total_cost_cents} cents`, 'bg-purple-100');
                    }
                    
                    addActivity('📄 Live transcript yüklendi');
                }
            } catch (error) {
                console.error('Live transcript display error:', error);
            }
        }

        // Arama durumunu kontrol et
        async function checkCallStatus() {
            if (!callId) return;
            
            try {
                const response = await fetch(`/api/conversation?action=status&call_id=${callId}`);
                const data = await response.json();
                
                if (data.success) {
                    const status = data.status;
                    const details = data.details;
                    
                    console.log('📊 Call Status:', status, details);
                    
                    // Durum güncellemesi
                    updateCallStatusUI(status, details);
                    
                    // Eğer arama tamamlandıysa transcript'i al
                    if (status === 'done' || status === 'completed') {
                        if (isCallActive) {
                            addMessage('Sistem', '✅ Arama tamamlandı. Transcript yükleniyor...');
                            await loadConversationTranscript();
                            endCallSimulation();
                        }
                    } else if (status === 'failed') {
                        if (isCallActive) {
                            addMessage('Sistem', '❌ Arama başarısız oldu.');
                            endCallSimulation();
                        }
                    }
                    
                } else {
                    console.warn('Status check failed:', data.error);
                }
                
            } catch (error) {
                console.error('Status check error:', error);
            }
        }

        // Arama durumu UI güncellemesi
        function updateCallStatusUI(status, details) {
            const statusElement = document.getElementById('call-status');
            let statusText = '';
            let statusColor = 'green';
            
            switch (status) {
                case 'pending':
                    statusText = '🔄 Arama başlatılıyor...';
                    statusColor = 'yellow';
                    break;
                case 'dialing':
                    statusText = '📞 Aranıyor...';
                    statusColor = 'blue';
                    break;
                case 'in-progress':
                case 'active':
                    statusText = '🗣️ Konuşuyor...';
                    statusColor = 'green';
                    break;
                case 'done':
                case 'completed':
                    statusText = '✅ Arama tamamlandı';
                    statusColor = 'gray';
                    break;
                case 'failed':
                    statusText = '❌ Arama başarısız';
                    statusColor = 'red';
                    break;
                default:
                    statusText = `🔴 Arama aktif (${status})`;
                    statusColor = 'green';
            }
            
            if (isCallActive) {
                statusElement.innerHTML = `
                    <div class="inline-flex items-center justify-center w-20 h-20 bg-${statusColor}-100 rounded-full mb-3 relative">
                        <div class="absolute inset-0 rounded-full bg-${statusColor}-400 opacity-25 pulse-ring"></div>
                        <i class="fas fa-phone text-${statusColor}-600 text-2xl"></i>
                    </div>
                    <p class="text-${statusColor}-600 font-medium">${statusText}</p>
                `;
                
                addActivity(`📊 Durum: ${statusText}`);
            }
        }

        // Conversation transcript'ini yükle
        async function loadConversationTranscript() {
            if (!conversationId) return;
            
            try {
                // Conversation ID'den "batch_" prefix'ini temizle
                const cleanConversationId = conversationId.replace('batch_', '');
                
                const response = await fetch(`/api/conversation?action=transcript&conversation_id=${cleanConversationId}`);
                const data = await response.json();
                
                if (data.success && data.transcript) {
                    const transcript = data.transcript;
                    
                    console.log('📝 Transcript:', transcript);
                    
                    // Transcript'i chat arayüzüne ekle
                    if (transcript.transcript && transcript.transcript.length > 0) {
                        addMessage('Sistem', '📝 Konuşma Kaydı:');
                        
                        transcript.transcript.forEach(entry => {
                            const role = entry.role === 'user' ? 'Müşteri' : 'Ayşe';
                            const time = entry.time_in_call_secs ? `(${Math.floor(entry.time_in_call_secs/60)}:${(entry.time_in_call_secs%60).toString().padStart(2, '0')})` : '';
                            addMessage(role, `${entry.message} ${time}`);
                        });
                        
                        // Metadata bilgilerini göster
                        if (transcript.metadata) {
                            const duration = transcript.metadata.call_duration_secs;
                            const startTime = new Date(transcript.metadata.start_time_unix_secs * 1000).toLocaleString('tr-TR');
                            
                            addMessage('Sistem', `⏱️ Arama süresi: ${Math.floor(duration/60)}:${(duration%60).toString().padStart(2, '0')}`);
                            addMessage('Sistem', `📅 Başlangıç: ${startTime}`);
                        }
                        
                        addActivity('📝 Transcript yüklendi ✓');
                    } else {
                        addMessage('Sistem', '⚠️ Bu arama için transcript bulunamadı.');
                    }
                    
                } else {
                    console.warn('Transcript fetch failed:', data.error);
                    addMessage('Sistem', '⚠️ Transcript yüklenemedi: ' + (data.error || 'Bilinmeyen hata'));
                }
                
            } catch (error) {
                console.error('Transcript load error:', error);
                addMessage('Sistem', '❌ Transcript yükleme hatası: ' + error.message);
            }
        }

        // Load knowledge base status
        async function loadKnowledgeBaseStatus() {
            try {
                const response = await fetch('/api/knowledge/list');
                const data = await response.json();
                
                const count = data.knowledge_base ? data.knowledge_base.length : 0;
                document.getElementById('knowledge-count').innerHTML = count + ' öğe';
                
                if (count === 0) {
                    showSystemAlert('Bilgi tabanı boş. Varsayılan içerikleri yüklemek için "Bilgi Tabanını Yükle" butonuna tıklayın.', 'warning');
                }
                
            } catch (error) {
                document.getElementById('knowledge-count').innerHTML = 'Hata';
                addActivity('⚠️ Bilgi tabanı durumu alınamadı');
            }
        }

        // Knowledge Base Modal Functions
        let knowledgeBaseData = [];
        let filteredKnowledgeBase = [];

        async function showKnowledgeBaseModal() {
            document.getElementById('knowledge-base-modal').classList.remove('hidden');
            await loadKnowledgeBaseData();
        }

        function hideKnowledgeBaseModal() {
            document.getElementById('knowledge-base-modal').classList.add('hidden');
            hideAddContentForm();
            hideEditContentForm();
        }

        async function loadKnowledgeBaseData() {
            try {
                const response = await fetch('/api/knowledge/list');
                const data = await response.json();
                
                if (data.success) {
                    knowledgeBaseData = data.knowledge_base || [];
                    filteredKnowledgeBase = [...knowledgeBaseData];
                    
                    // Load categories
                    loadCategories(data.categories || {});
                    
                    // Display content
                    displayKnowledgeContent();
                    displayCategorySummary(data.categories || {});
                } else {
                    alert('Bilgi veritabanı yüklenemedi: ' + (data.error || 'Bilinmeyen hata'));
                }
                
            } catch (error) {
                alert('Bilgi veritabanı yükleme hatası: ' + error.message);
            }
        }

        function loadCategories(categories) {
            const categoryFilter = document.getElementById('category-filter');
            const contentCategory = document.getElementById('content-category');
            
            // Clear existing options
            categoryFilter.innerHTML = '<option value="">Tüm Kategoriler</option>';
            contentCategory.innerHTML = '<option value="">Kategori Seçin</option>';
            
            // Add categories
            Object.keys(categories).forEach(key => {
                const option1 = new Option(categories[key], key);
                const option2 = new Option(categories[key], key);
                categoryFilter.add(option1);
                contentCategory.add(option2);
            });
        }

        function displayCategorySummary(categories) {
            const categoriesDiv = document.getElementById('knowledge-categories');
            let html = '';
            
            // Count items per category
            const categoryCounts = {};
            knowledgeBaseData.forEach(item => {
                categoryCounts[item.category] = (categoryCounts[item.category] || 0) + 1;
            });
            
            // Display categories with counts
            Object.keys(categories).forEach(key => {
                const count = categoryCounts[key] || 0;
                html += `
                    <div class="flex items-center justify-between p-2 bg-white rounded border ${count > 0 ? 'border-green-200' : 'border-gray-200'}">
                        <span class="text-sm font-medium">${categories[key]}</span>
                        <span class="text-xs px-2 py-1 rounded-full ${count > 0 ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-600'}">${count}</span>
                    </div>
                `;
            });
            
            categoriesDiv.innerHTML = html;
        }

        function displayKnowledgeContent() {
            const contentDiv = document.getElementById('knowledge-content-list');
            
            if (filteredKnowledgeBase.length === 0) {
                contentDiv.innerHTML = `
                    <div class="text-center py-8 text-gray-500">
                        <i class="fas fa-search text-4xl mb-4"></i>
                        <p>Hiç içerik bulunamadı</p>
                    </div>
                `;
                return;
            }
            
            let html = '<div class="space-y-4">';
            
            filteredKnowledgeBase.forEach(item => {
                const createdDate = new Date(item.created_at).toLocaleDateString('tr-TR');
                const truncatedContent = item.content.length > 150 
                    ? item.content.substring(0, 150) + '...' 
                    : item.content;
                
                html += `
                    <div class="border rounded-lg p-4 hover:shadow-md transition-shadow">
                        <div class="flex items-start justify-between mb-2">
                            <h5 class="font-semibold text-lg">${item.title}</h5>
                            <div class="flex space-x-2">
                                <button onclick="editKnowledgeItem('${item.content_id}')" 
                                        class="px-2 py-1 text-xs bg-blue-100 text-blue-600 rounded hover:bg-blue-200">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button onclick="deleteKnowledgeItem('${item.content_id}', '${item.title}')" 
                                        class="px-2 py-1 text-xs bg-red-100 text-red-600 rounded hover:bg-red-200">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </div>
                        
                        <div class="text-sm text-gray-600 mb-2">
                            <span class="inline-flex items-center px-2 py-1 rounded-full text-xs bg-gray-100">
                                ${item.category}
                            </span>
                            <span class="ml-2">${createdDate}</span>
                            ${item.source ? `<span class="ml-2 text-blue-600">${item.source}</span>` : ''}
                        </div>
                        
                        <p class="text-gray-700 text-sm">${truncatedContent}</p>
                        
                        ${item.elevenlabs_id ? 
                            '<div class="mt-2 text-xs text-green-600"><i class="fas fa-check-circle mr-1"></i>Elevenlabs\'ta senkronize</div>' : 
                            '<div class="mt-2 text-xs text-orange-600"><i class="fas fa-exclamation-triangle mr-1"></i>Elevenlabs senkronizasyonu bekliyor</div>'
                        }
                    </div>
                `;
            });
            
            html += '</div>';
            contentDiv.innerHTML = html;
        }

        function filterKnowledgeBase() {
            const category = document.getElementById('category-filter').value;
            const searchTerm = document.getElementById('knowledge-search').value.toLowerCase();
            
            filteredKnowledgeBase = knowledgeBaseData.filter(item => {
                const matchesCategory = !category || item.category === category;
                const matchesSearch = !searchTerm || 
                    item.title.toLowerCase().includes(searchTerm) || 
                    item.content.toLowerCase().includes(searchTerm);
                
                return matchesCategory && matchesSearch;
            });
            
            displayKnowledgeContent();
        }

        function searchKnowledgeBase() {
            filterKnowledgeBase();
        }

        function showAddContentForm() {
            document.getElementById('knowledge-content-list').classList.add('hidden');
            document.getElementById('edit-content-form').classList.add('hidden');
            document.getElementById('add-content-form').classList.remove('hidden');
            
            // Clear form
            document.getElementById('content-title').value = '';
            document.getElementById('content-category').value = '';
            document.getElementById('content-text').value = '';
        }

        function hideAddContentForm() {
            document.getElementById('add-content-form').classList.add('hidden');
            document.getElementById('knowledge-content-list').classList.remove('hidden');
        }

        function showEditContentForm() {
            document.getElementById('knowledge-content-list').classList.add('hidden');
            document.getElementById('add-content-form').classList.add('hidden');
            document.getElementById('edit-content-form').classList.remove('hidden');
        }

        function hideEditContentForm() {
            document.getElementById('edit-content-form').classList.add('hidden');
            document.getElementById('knowledge-content-list').classList.remove('hidden');
        }

        async function addKnowledgeContent(event) {
            event.preventDefault();
            
            const title = document.getElementById('content-title').value;
            const category = document.getElementById('content-category').value;
            const content = document.getElementById('content-text').value;
            
            try {
                const response = await fetch('/api/knowledge', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'add',
                        title: title,
                        category: category,
                        content: content,
                        type: 'text'
                    })
                });

                const data = await response.json();
                
                if (data.success) {
                    alert('İçerik başarıyla eklendi!');
                    hideAddContentForm();
                    await loadKnowledgeBaseData();
                    await loadKnowledgeBaseStatus(); // Update main dashboard
                } else {
                    alert('İçerik eklenemedi: ' + (data.error || 'Bilinmeyen hata'));
                }
                
            } catch (error) {
                alert('İçerik ekleme hatası: ' + error.message);
            }
        }

        async function editKnowledgeItem(contentId) {
            const item = knowledgeBaseData.find(item => item.content_id === contentId);
            if (!item) {
                alert('İçerik bulunamadı');
                return;
            }
            
            // Fill edit form
            document.getElementById('edit-content-id').value = contentId;
            document.getElementById('edit-content-title').value = item.title;
            document.getElementById('edit-content-text').value = item.content;
            
            showEditContentForm();
        }

        async function updateKnowledgeContent(event) {
            event.preventDefault();
            
            const contentId = document.getElementById('edit-content-id').value;
            const title = document.getElementById('edit-content-title').value;
            const content = document.getElementById('edit-content-text').value;
            
            try {
                const response = await fetch('/api/knowledge', {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        content_id: contentId,
                        content: content
                    })
                });

                const data = await response.json();
                
                if (data.success) {
                    alert('İçerik başarıyla güncellendi!');
                    hideEditContentForm();
                    await loadKnowledgeBaseData();
                } else {
                    alert('İçerik güncellenemedi: ' + (data.error || 'Bilinmeyen hata'));
                }
                
            } catch (error) {
                alert('İçerik güncelleme hatası: ' + error.message);
            }
        }

        async function deleteKnowledgeItem(contentId, title) {
            if (!confirm(`"${title}" içeriğini silmek istediğinizden emin misiniz?`)) {
                return;
            }
            
            try {
                const response = await fetch('/api/knowledge', {
                    method: 'DELETE',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        content_id: contentId
                    })
                });

                const data = await response.json();
                
                if (data.success) {
                    alert('İçerik başarıyla silindi!');
                    await loadKnowledgeBaseData();
                    await loadKnowledgeBaseStatus(); // Update main dashboard
                } else {
                    alert('İçerik silinemedi: ' + (data.error || 'Bilinmeyen hata'));
                }
                
            } catch (error) {
                alert('İçerik silme hatası: ' + error.message);
            }
        }

        // Initialize knowledge base
        async function initializeKnowledgeBase() {
            if (confirm('Varsayılan bilgi tabanını yüklemek istediğinizden emin misiniz?')) {
                try {
                    const response = await fetch('/api/knowledge/initialize', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        }
                    });
                    
                    const data = await response.json();
                    
                    if (data.success) {
                        alert('✅ Bilgi tabanı başarıyla yüklendi!\n\n' + (data.message || 'Varsayılan içerikler eklendi.'));
                        loadKnowledgeBaseStatus();
                        addActivity('📚 ' + (data.message || 'Bilgi tabanı başlatıldı'));
                        hideSystemAlert();
                    } else {
                        alert('❌ Bilgi tabanı yüklenirken hata oluştu:\n\n' + (data.error || 'Bilinmeyen hata'));
                        addActivity('⚠️ Bilgi tabanı yükleme hatası: ' + (data.error || 'Bilinmeyen hata'));
                    }
                } catch (error) {
                    alert('❌ Bilgi tabanı yüklenirken hata oluştu');
                    addActivity('⚠️ Bilgi tabanı yükleme hatası');
                }
            }
        }

        // Show/hide system alert
        function showSystemAlert(message, type = 'warning') {
            const alert = document.getElementById('system-status-alert');
            const messageEl = document.getElementById('status-message');
            const alertContainer = alert.querySelector('div');
            
            messageEl.textContent = message;
            
            // Update alert style based on type
            alertContainer.className = type === 'success' 
                ? 'bg-green-50 border-l-4 border-green-400 p-4 rounded'
                : type === 'error'
                ? 'bg-red-50 border-l-4 border-red-400 p-4 rounded'
                : 'bg-yellow-50 border-l-4 border-yellow-400 p-4 rounded';
            
            const icon = alertContainer.querySelector('i');
            icon.className = type === 'success'
                ? 'fas fa-check-circle text-green-400'
                : type === 'error'
                ? 'fas fa-exclamation-circle text-red-400'
                : 'fas fa-exclamation-triangle text-yellow-400';
            
            messageEl.className = type === 'success'
                ? 'text-sm text-green-700'
                : type === 'error'
                ? 'text-sm text-red-700'
                : 'text-sm text-yellow-700';
            
            alert.classList.remove('hidden');
            
            // Auto hide after specified time
            if (type !== 'error') {
                setTimeout(hideSystemAlert, type === 'success' ? 5000 : 10000);
            }
        }

        function hideSystemAlert() {
            document.getElementById('system-status-alert').classList.add('hidden');
        }

        // Add activity
        function addActivity(activity) {
            const container = document.getElementById('recent-activity');
            const now = new Date().toLocaleTimeString('tr-TR');
            
            const activityDiv = document.createElement('div');
            activityDiv.className = 'flex items-center';
            activityDiv.innerHTML = `
                <i class="fas fa-circle text-blue-400 text-xs mr-2"></i>
                ${activity}
                <span class="text-xs text-gray-400 ml-auto">${now}</span>
            `;
            
            container.insertBefore(activityDiv, container.firstChild);
            
            // Keep only last 8 activities
            while (container.children.length > 8) {
                container.removeChild(container.lastChild);
            }
        }

        // Load voice list
        async function loadVoiceList() {
            try {
                const response = await fetch('/api/voices/list');
                const data = await response.json();
                
                if (data.success) {
                    const selector = document.getElementById('voice-selector');
                    selector.innerHTML = '<option value="">Ses seçin...</option>';
                    
                    data.voices.forEach(voice => {
                        const option = document.createElement('option');
                        option.value = voice.voice_id;
                        option.textContent = `${voice.name} (${voice.language})`;
                        option.dataset.language = voice.language;
                        option.dataset.category = voice.category;
                        option.dataset.description = voice.description;
                        
                        if (voice.voice_id === data.current_voice) {
                            option.selected = true;
                            showVoiceInfo(voice);
                        }
                        
                        selector.appendChild(option);
                    });
                    
                    addActivity(`🎤 ${data.total_count} ses seçeneği yüklendi`);
                } else {
                    document.getElementById('voice-selector').innerHTML = '<option value="">Hata: Sesler yüklenemedi</option>';
                }
            } catch (error) {
                console.error('Voice list error:', error);
                document.getElementById('voice-selector').innerHTML = '<option value="">Bağlantı hatası</option>';
                addActivity('⚠️ Ses listesi yüklenemedi');
            }
        }

        // Select voice
        async function selectVoice() {
            const selector = document.getElementById('voice-selector');
            const selectedOption = selector.options[selector.selectedIndex];
            
            if (!selectedOption.value) {
                document.getElementById('voice-info').classList.add('hidden');
                return;
            }
            
            const voiceInfo = {
                voice_id: selectedOption.value,
                name: selectedOption.textContent.split(' (')[0],
                language: selectedOption.dataset.language,
                category: selectedOption.dataset.category,
                description: selectedOption.dataset.description
            };
            
            showVoiceInfo(voiceInfo);
            
            try {
                const response = await fetch('/api/voices/select', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        voice_id: voiceInfo.voice_id,
                        voice_name: voiceInfo.name
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    addActivity(`✅ Ses değiştirildi: ${data.voice_name}`);
                    // Restart API test to update voice status
                    setTimeout(testAPIConnection, 1000);
                } else {
                    alert('Ses değiştirilemedi: ' + data.error);
                }
            } catch (error) {
                console.error('Voice selection error:', error);
                alert('Ses seçimi hatası: ' + error.message);
            }
        }

        // Show voice info
        function showVoiceInfo(voice) {
            document.getElementById('voice-language').textContent = voice.language || 'Bilinmiyor';
            document.getElementById('voice-category').textContent = voice.category || 'Bilinmiyor';
            document.getElementById('voice-info').classList.remove('hidden');
        }

        // Test voice
        function testVoice() {
            const selector = document.getElementById('voice-selector');
            const selectedOption = selector.options[selector.selectedIndex];
            
            if (!selectedOption.value) {
                alert('Lütfen önce bir ses seçin');
                return;
            }
            
            const voiceName = selectedOption.textContent.split(' (')[0];
            addActivity(`🎵 Ses testi: ${voiceName}`);
            
            // Simulate voice test (In real implementation, this would play a sample)
            setTimeout(() => {
                addActivity(`✅ Ses testi tamamlandı: ${voiceName}`);
            }, 2000);
        }
    </script>
</body>
</html> 